﻿/**
 * Loading screen for multiply portals and error page. Views handling module.
 * @author Fedotov Dmytro <bas.jsdev@gmail.com>
 */

'use strict';

app.view = (function ( global ) {
    var module = {};


    /**
     * Page initialization.
     */
    module.init = function () {
        // set webkit size
        global.moveTo(0, 0);
        window.resizeTo(EMULATION ? window.outerWidth : screen.width, EMULATION ? window.outerHeight : screen.height);

        global.CMODAL_IMG_PATH = configuration.newRemoteControl ? PATH_IMG_SYSTEM + 'buttons/new/' : PATH_IMG_SYSTEM + 'buttons/old/';

        /**
         * toggle stand by mode
         */
        module.standByMode = function () {
            var standByStatus = !gSTB.GetStandByStatus(),
                standByEnv = JSON.parse(gSTB.GetEnv(JSON.stringify({varList: ['standbyLedLevel', 'standByMode']}))).result,
                standByMode;

            if ( standByEnv.standByMode !== '' ) {
                app.data.environmentData.standByMode = standByEnv.standByMode;
            }

            if ( standByEnv.standbyLedLevel !== '' ) {
                app.data.environmentData.standbyLedLevel = standByEnv.standbyLedLevel;
            }

            standByMode = parseInt(app.data.environmentData.standByMode, 10)

            echo('PORTALS_LOADER.standByStatus=' + standByStatus);
            // check stand by mode trigger
            if ( gSTB.StandByMode !== standByMode ) {
                gSTB.StandByMode = standByMode;
            }
            // deep standBy mode
            if ( app.standByMode === 3 ) {
                gSTB.SetLedIndicatorMode(2);
                gSTB.StandBy(standByStatus);
                gSTB.SetLedIndicatorMode(1);
                return;
            }
            // active standby mode
            gSTB.StandBy(standByStatus);
            if ( standByStatus ) {
                gSTB.SetLedIndicatorMode(2);
            } else {
                gSTB.SetLedIndicatorMode(1);
            }
        };

        stbEvent.bind('window.reload', function () {
            document.body.style.display = 'none';
            stbStorage.removeItem(getWindowKey(WINDOWS.PORTALS_LOADER));
            window.location.reload();
        });

        stbEvent.bind('portal.standbyMode', function () {
            echo('got portal.standbyMode');
            module.standByMode();
        });
    };


    /**
     * Set special item in list
     * @param {Object} data portals list
     * @returns {Object} data modified portal list with new items
     */
    module.setServiceItems = function ( data ) {
        var items                 = [],
            serviceItemsNumber    = 0,
            hasServiceSettingItem = false;

        // set 'try again' as first item (but don't show it for blocked urls)
        if ( !app.pageUrl.queryKey.bootmedia && app.pageUrl.queryKey.blocked !== 'true' && app.pageUrl.query ) {
            items.push({
                name: _('Try again'),
                url: app.pageUrl.queryKey.url ? decodeURIComponent(app.pageUrl.queryKey.url) : decodeURIComponent(app.pageUrl.query),
                enable: true
            });
            serviceItemsNumber++;
        }
        // Set inner portal as next item (if allowed)
        if ( RULES.allowInnerPortal ) {
            items.push({
                name: _('Embedded portal'),
                url: PATH_ROOT + 'services.html',
                enable: true,
                servItem: true
            });
            serviceItemsNumber++;
        }
        // portals
        (data.portals || []).forEach(function ( item ) {
            items.push(item);
        });
        // Set service menu as next item (if allowed)
        if ( RULES['page404/allowSystemSettings'] && app.isItErrorPage || RULES['portalsLoader/allowSystemSettings'] && !app.isItErrorPage ) {
            items.push({
                name: _('System settings'),
                url: PATH_SYSTEM + 'settings/index.html',
                enable: true,
                servItem: true
            });
            hasServiceSettingItem = true;
        }

        return {
            itemsArray: items,
            serviceItemsNumber: serviceItemsNumber,
            hasServiceSettingItem: hasServiceSettingItem
        };
    };


    /**
     * Set corresponding icon style
     * @param {string} url current item url
     * @param {string} name current item name
     * @returns {string} iconStyle CSS class name for specific icon.
     */
    module.getPortalType = function ( url, name ) {
        var urlByParts = parseUri(url),
            iconClass;

        // settings page
        if ( 'file://' + urlByParts.path === PATH_SYSTEM + 'settings/index.html' ) {
            iconClass = 'settings';
        } else
        // refresh action
        if ( name === _('Try again') ) {
            iconClass = 'refresh';
        } else
        // use internet for portal loading
        if ( urlByParts.protocol === 'http' || urlByParts.protocol === 'https' ) {
            iconClass = 'internet';
        } else
        // use path 'file:///home/...page.html' or '/home/web/...page.html' or '/media/...page.html'
        if ( urlByParts.protocol.indexOf('file') !== -1 || urlByParts.path.indexOf(PATH_ROOT) !== -1 || urlByParts.path.indexOf(PATH_MEDIA) !== -1 ) {
            iconClass = 'home';
        } else {
            iconClass = 'local'; // use local network for loading
        }
        return iconClass;
    };

    module.checkForUpdate = function ( callback ) {
        var request = new XMLHttpRequest(),
            hashData = '',
            environment = app.data.environmentData,
            currTimeStamp = Math.round(new Date().getTime() / 1000),
            AUTO_UPDATE = 0,
            UPDATE_WITH_CONFIRMATION = 1;

        if ( environment.autoUpdateCondition === AUTO_UPDATE || environment.autoUpdateCondition === UPDATE_WITH_CONFIRMATION ) {
            request.open('GET', (environment.autoupdateURL || configuration.url.updateList) + '?mac=' + gSTB.GetDeviceMacAddress(), true);

            request.onreadystatechange = function () {
                var updateList,
                    versions = [],
                    updateModal;

                if ( request.readyState === 4 && request.status === 200 ) {
                    try {
                        updateList = JSON.parse(request.responseText);
                    } catch ( error ) {
                        console.log('UpdateModal: cant\'t parse list with images for update', error);
                        updateList = [];
                    }

                    updateList.forEach(function ( image ) {
                        if (
                            image.type === 'release'
                            && (
                                Date.parse(image.date) > environment.ImageDate.getTime()
                                || isNaN(environment.ImageDate.getTime())
                            )
                        ) {
                            image.title = image.name + ' <span class="grey">(' + new Date(image.date).toDateString() + ') </span>';
                            versions.unshift(image);
                        }
                    });

                    if ( versions.length === 0 ) {
                        echo('UpdateModal: Got images list from server. No need for update.');
                        callback();
                        return;
                    }

                    updateModal = new CUpdateModal(null, {
                        images: versions,
                        auto: environment.autoUpdateCondition === AUTO_UPDATE,
                        select: false,
                        header_text: _('Software autoupdate'),
                        info: true,
                        log: true,
                        events: {
                            onStart: function () {
                                if ( environment.autoUpdateCondition === 0 ) {
                                    if ( this.images !== undefined ) {
                                        var image;

                                        if ( typeof this.select === 'object' ) {
                                            image = this.select.GetSelected();
                                        } else {
                                            image = this.images[0];
                                        }

                                        if ( Date.parse(image.date) < environment.ImageDate.getTime() ) {
                                            gSTB.SetEnv(JSON.stringify({autoupdate_cond: 1}));
                                        }
                                    }
                                }
                            },
                            onError: function () {
                                echo('UpdateModal: error happened. Prevent cyclic auto update. Set autoupdate_cond to:' + environment.autoUpdateCondition);
                                gSTB.SetEnv(JSON.stringify({autoupdate_cond: environment.autoUpdateCondition}));
                                // prevent cyclic auto update
                                this.auto = false;
                                callback();
                            },
                            onHide: function () {
                                echo('UpdateModal: closed');
                                // user declined update, continue default loading behaviour
                                callback();
                            }
                        }
                    });

                    updateModal.Show(true);
                }
            };

            // modelNameEx + currentVersion + serialNumber + macAddr + currentLang + portalUrl + porta1 + portal2 + userAgent + current_t_wo_sec
            hashData += (gSTB.GetDeviceModelExt ? gSTB.GetDeviceModelExt(true) : gSTB.GetDeviceModel());
            hashData += gSTB.GetDeviceImageDesc() + gSTB.GetDeviceSerialNumber() + gSTB.GetDeviceMacAddress();
            hashData += (environment.language || 'en') + location.href + (environment.portal1 || '') + (environment.portal2 || '');
            hashData += navigator.userAgent + (currTimeStamp - currTimeStamp % 60);
            // echo(hashData);
            request.setRequestHeader('X-CurrentVersion', gSTB.GetDeviceImageDesc());
            request.setRequestHeader('X-SerialNumber', gSTB.GetDeviceSerialNumber());
            request.setRequestHeader('X-MacAddress', gSTB.GetDeviceMacAddress());
            request.setRequestHeader('X-ModelNameEx', gSTB.GetDeviceModelExt ? gSTB.GetDeviceModelExt(true) : gSTB.GetDeviceModel());
            request.setRequestHeader('X-Language', environment.language || 'en');
            request.setRequestHeader('X-PortalUrl', location.href);
            request.setRequestHeader('X-Hash', gSTB.GetHashVersion1(hashData, ''));
            environment.portal1 && request.setRequestHeader('X-Portal1', environment.portal1);
            environment.portal2 && request.setRequestHeader('X-Portal2', environment.portal2);

            request.send();
        } else {
            echo('UpdateModal: update condition set to 2, no need to update');
            // user don't want to update automatically (it can be changed in user settings)
            callback();
        }
    };


    // export
    return module;
})(window);
